<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Event;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

class UpdateEventStatus extends Command
{
    // Command signature to run via artisan (e.g., php artisan events:update-status)
    protected $signature = 'events:update-status';

    // Description for the command
    protected $description = 'Update event statuses to D if end time has passed';

    public function __construct()
    {
        parent::__construct();
    }

    public function handle()
    {
        // Log the start of the process for debugging
        Log::info('Starting event status update process.');

        // Fetch all events with status 'A' (Approved)
        $events = Event::where('E_Status', 'A')->get();

        // If no events are found, log and exit
        if ($events->isEmpty()) {
            Log::info('No events with status "A" found.');
            $this->info('No events to update.');
            return;
        }

        // Loop through each event
        foreach ($events as $event) {
            // Parse the end date and time in IST (thanks to config/app.php timezone)
            $endDateTime = Carbon::parse($event->E_EndDate . ' ' . $event->E_EndTime);
            $now = Carbon::now(); // Current time in IST

            // Log the event details for debugging
            Log::info("Checking event ID: {$event->E_Id}, End DateTime: {$endDateTime}, Current Time: {$now}");

            // Check if the end time has passed
            if ($endDateTime->isPast()) {
                // Update the status to 'D' (Done)
                $event->update(['E_Status' => 'D']);
                Log::info("Updated event ID: {$event->E_Id} to status 'D'");
                $this->info("Updated event {$event->E_Id} to status 'D'");
            } else {
                Log::info("Event ID: {$event->E_Id} has not ended yet.");
            }
        }

        // Log completion
        Log::info('Event status update completed.');
        $this->info('Event status update completed.');
    }
}
