<?php

namespace App\Http\Controllers;

use App\Models\AdditionalPeople;
use App\Models\BookEvent;
use App\Models\Event;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class BookEventController extends Controller
{
    public function index()
    {
        try {
            $bookings = BookEvent::with(['event', 'user', 'additionalPeople'])->get();

            // Transform the response to format additional_people and user data cleanly
            $formattedBookings = $bookings->map(function ($booking) {
                return [
                    'BE_Id' => $booking->BE_Id,
                    'BE_Name' => $booking->BE_Name,
                    'BE_Email' => $booking->BE_Email,
                    'BE_Mobile' => $booking->BE_Mobile,
                    'E_Id' => $booking->E_Id,
                    'U_Id' => $booking->U_Id,
                    'Comm_Id' => $booking->Comm_Id,
                    'BE_Code' => $booking->BE_Code,
                    'created_at' => $booking->created_at,
                    'updated_at' => $booking->updated_at,
                    'event' => $booking->event,
                    'user' => [
                        'U_Id' => $booking->user->U_Id,
                        'U_Name' => $booking->user->U_Name,
                        'U_Email' => $booking->user->U_Email,
                        'U_Mobile' => $booking->user->U_Mobile,
                        'Role_Id' => $booking->user->Role_Id
                    ],
                    'additional_people' => $booking->additionalPeople->map(function ($person) {
                        return [
                            'name' => $person->name,
                            'email' => $person->email,
                            'mobile' => $person->mobile
                        ];
                    })->toArray()
                ];
            });

            return response()->json([
                'status' => 'success',
                'data' => $formattedBookings
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch bookings',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'BE_Name' => 'required|string',
            'BE_Email' => 'required|email',
            'BE_Mobile' => 'required|string',
            'E_Id' => 'required|exists:events,E_Id',
            'U_Id' => 'required|exists:user_masters,U_Id',
            'additional_people' => 'nullable|array',
            'additional_people.*.name' => 'required_with:additional_people|string',
            'additional_people.*.email' => 'required_with:additional_people|email',
            'additional_people.*.mobile' => 'required_with:additional_people|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => 'Validation failed',
                'messages' => $validator->errors()
            ], 422);
        }
        $event = Event::find($request->E_Id);
        $user = UserMaster::find($request->U_Id);

        if (!$event || !$user) {
            return response()->json(['error' => 'Event or User not found'], 404);
        }

        // Count total people: 1 for the registered user + number of additional people
        $additionalPeople = $request->additional_people ?? [];
        $totalPeople = 1 + count($additionalPeople);

        // Check if event has enough capacity
        $remainingCapacity = $event->E_Capacity - $event->E_Booked_Count;
        if ($totalPeople > $remainingCapacity) {
            return response()->json([
                'error' => "Event capacity exceeded. Remaining capacity: $remainingCapacity, Requested: $totalPeople"
            ], 400);
        }

        // Start a transaction
        DB::beginTransaction();
        try {
            // Generate a unique 6-character alphanumeric BE_Code
            do {
                $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
                $beCode = '';
                for ($i = 0; $i < 6; $i++) {
                    $beCode .= $characters[rand(0, strlen($characters) - 1)];
                }
            } while (BookEvent::where('BE_Code', $beCode)->exists()); // Check for uniqueness

            // Create booking with 6-character alphanumeric BE_Code
            $booking = BookEvent::create([
                'BE_Name' => $request->BE_Name,
                'BE_Email' => $request->BE_Email,
                'BE_Mobile' => $request->BE_Mobile,
                'E_Id' => $request->E_Id,
                'U_Id' => $request->U_Id,
                'BE_Code' => $beCode,
            ]);

            // Save additional people in the additional_people table
            foreach ($additionalPeople as $person) {
                AdditionalPeople::create([
                    'BE_Id' => $booking->BE_Id,
                    'name' => $person['name'],
                    'email' => $person['email'],
                    'mobile' => $person['mobile'],
                ]);
            }

            // Update booked count in events table
            $event->E_Booked_Count += $totalPeople;
            $event->save();

            DB::commit();

            // Load relationships for response
            $booking->load(['event', 'user', 'additionalPeople']);
            $formattedBooking = [
                'BE_Id' => $booking->BE_Id,
                'BE_Name' => $booking->BE_Name,
                'BE_Email' => $booking->BE_Email,
                'BE_Mobile' => $booking->BE_Mobile,
                'E_Id' => $booking->E_Id,
                'U_Id' => $booking->U_Id,
                'BE_Code' => $booking->BE_Code,
                'event' => $booking->event,
                'user' => $booking->user,
                'additional_people' => $booking->additionalPeople->map(function ($person) {
                    return [
                        'name' => $person->name,
                        'email' => $person->email,
                        'mobile' => $person->mobile
                    ];
                })->toArray()
            ];

            return response()->json([
                'message' => "Event booked successfully for $totalPeople people (1 registered + " . count($additionalPeople) . " additional)",
                'booking' => $formattedBooking
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Failed to book event: ' . $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'BE_Name' => 'nullable|string',
                'BE_Email' => 'nullable|email',
                'BE_Mobile' => 'nullable|digits:10',
                'E_Id' => 'nullable|exists:events,E_Id',
                'U_Id' => 'nullable|integer',
                'Comm_Id' => 'nullable|integer',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $booking = BookEvent::find($id);
            if (!$booking) {
                return response()->json(['error' => 'Booking not found.'], 404);
            }

            // Conditional checks only if fields are present
            if ($request->has('U_Id')) {
                $user = UserMaster::where('U_Id', $request->U_Id)->first();
                if (!$user) {
                    return response()->json(['error' => 'User does not exist.'], 422);
                }
            } else {
                $user = UserMaster::where('U_Id', $booking->U_Id)->first();
            }

            if ($request->has('E_Id')) {
                $event = Event::find($request->E_Id);
                if (!$event) {
                    return response()->json(['error' => 'Event not found.'], 422);
                }

                if ($booking->E_Id != $request->E_Id) {
                    $currentBookings = BookEvent::where('E_Id', $request->E_Id)->count();
                    if ($currentBookings >= $event->E_Capacity) {
                        return response()->json(['error' => 'Booking limit reached for this event.'], 422);
                    }
                }
            }

            // Only update fields that are present
            $dataToUpdate = $request->only(['BE_Name', 'BE_Email', 'BE_Mobile', 'E_Id', 'U_Id', 'Comm_Id']);
            $booking->update($dataToUpdate);

            // Load relationships for response
            $booking->load(['event', 'user', 'additionalPeople']);
            $formattedBooking = [
                'BE_Id' => $booking->BE_Id,
                'BE_Name' => $booking->BE_Name,
                'BE_Email' => $booking->BE_Email,
                'BE_Mobile' => $booking->BE_Mobile,
                'E_Id' => $booking->E_Id,
                'U_Id' => $booking->U_Id,
                'BE_Code' => $booking->BE_Code,
                'created_at' => $booking->created_at,
                'updated_at' => $booking->updated_at,
                'event' => $booking->event,
                'user' => $booking->user,

                'additional_people' => $booking->additionalPeople->map(function ($person) {
                    return [
                        'name' => $person->name,
                        'email' => $person->email,
                        'mobile' => $person->mobile
                    ];
                })->toArray()
            ];

            return response()->json([
                'message' => 'Event booking updated successfully.',
                'data' => $formattedBooking
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        try {
            $booking = BookEvent::with(['event', 'user', 'additionalPeople'])->find($id);

            if (!$booking) {
                return response()->json(['status' => 'error', 'message' => 'Booking not found'], 404);
            }

            // Format the response
            $formattedBooking = [
                'BE_Id' => $booking->BE_Id,
                'BE_Name' => $booking->BE_Name,
                'BE_Email' => $booking->BE_Email,
                'BE_Mobile' => $booking->BE_Mobile,
                'E_Id' => $booking->E_Id,
                'U_Id' => $booking->U_Id,
                'Comm_Id' => $booking->Comm_Id,
                'BE_Code' => $booking->BE_Code,
                'created_at' => $booking->created_at,
                'updated_at' => $booking->updated_at,
                'event' => $booking->event,
                'user' => [
                    'U_Id' => $booking->user->U_Id,
                    'U_Name' => $booking->user->U_Name,
                    'U_Email' => $booking->user->U_Email,
                    'U_Mobile' => $booking->user->U_Mobile,
                    'Role_Id' => $booking->user->Role_Id
                ],
                'additional_people' => $booking->additionalPeople->map(function ($person) {
                    return [
                        'name' => $person->name,
                        'email' => $person->email,
                        'mobile' => $person->mobile
                    ];
                })->toArray()
            ];

            return response()->json([
                'status' => 'success',
                'data' => $formattedBooking
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch booking',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
