<?php

namespace App\Http\Controllers;

use App\Models\Celebration;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CelebrationController extends Controller
{

    public function index()
    {
        try {
            // Get all celebrations with the related user data (eager loading)
            $celebrations = Celebration::with('user')->get();

            // Loop through each celebration and decode the N_Image field to remove any backslashes
            foreach ($celebrations as $celebration) {
                if (is_string($celebration->N_Image)) {
                    $celebration->N_Image = json_decode($celebration->N_Image, true);
                }
            }

            return response()->json(['status' => true, 'message' => 'Celebrations fetched successfully', 'data' => $celebrations], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'N_Title' => 'required|string',
                'U_Id' => 'required|exists:user_masters,U_Id',
                'N_Image' => 'nullable|array',
                'N_Image.*' => 'nullable|image|mimes:jpeg,jpg,png,gif,bmp,webp|max:20480',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
            }

            $user = UserMaster::where('U_Id', $request->U_Id)->first();

            // Determine N_Status
            $status = 'pending'; // Default for Role_Id = 3
            if ($user && $user->Role_Id == 2) {
                $status = 'approved'; // Direct post for Role_Id = 2
            }

            // Handle multiple image uploads
            $photoPaths = [];
            if ($request->hasFile('N_Image')) {
                foreach ($request->file('N_Image') as $image) {
                    if ($image->isValid()) {
                        $filename = time() . '_' . $image->getClientOriginalName();
                        $image->move(public_path('celebration/'), $filename);
                        $photoPaths[] = asset('celebration/' . $filename);
                    }
                }
            }

            $celebration = Celebration::create([
                'N_Title'       => $request->N_Title,
                'N_Category'    => $request->N_Category,
                'N_MaleName'    => $request->N_MaleName,
                'N_WomanName'   => $request->N_WomanName,
                'N_Date'        => $request->N_Date,
                'N_Age'         => $request->N_Age,
                'N_Years'       => $request->N_Years,
                'N_Mobile'      => $request->N_Mobile,
                'N_Email'       => $request->N_Email,
                'N_Address'     => $request->N_Address,
                'N_Description' => $request->N_Description,
                'N_Image'       => $photoPaths,
                'U_Id'          => $request->U_Id,
                'N_Status'      => $status
            ]);

            return response()->json(['status' => true, 'message' => 'Celebration created successfully', 'data' => $celebration], 201);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        try {
            // Find celebration by ID with the related user data (eager loading)
            $celebration = Celebration::with('user')->find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            // Decode the N_Image field to remove any backslashes
            if (is_string($celebration->N_Image)) {
                $celebration->N_Image = json_decode($celebration->N_Image, true);
            }

            return response()->json(['status' => true, 'message' => 'Celebration fetched successfully', 'data' => $celebration], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }


 public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'N_Title'     => 'nullable|string',
                'U_Id'        => 'nullable|exists:user_masters,U_Id',
                'N_Image'     => 'sometimes|array',
                'N_Image.*'   => 'sometimes|string',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
            }

            $celebration = Celebration::find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            // Prevent status change if already rejected
            if ($celebration->N_Status === 'rejected' && $request->has('N_Status')) {
                return response()->json(['status' => false, 'message' => 'Status cannot be changed after being rejected'], 400);
            }

            // Get existing images
            $existingImages = is_string($celebration->N_Image) ? json_decode($celebration->N_Image, true) : [];
            $existingImages = is_array($existingImages) ? $existingImages : [];

            // Final image array to be stored
            $finalImages = $existingImages;

            // Handle new base64 images
            if ($request->has('N_Image') && is_array($request->N_Image)) {
                foreach ($request->N_Image as $imageData) {
                    // Only handle new base64 strings
                    if (strpos($imageData, 'data:image') === 0) {
                        list($type, $imageData) = explode(';', $imageData);
                        list(, $imageData) = explode(',', $imageData);

                        $directory = public_path('celebration/');
                        if (!file_exists($directory)) {
                            mkdir($directory, 0777, true);
                        }

                        $imageName = time() . '_' . uniqid() . '.png';
                        file_put_contents($directory . $imageName, base64_decode($imageData));
                        $imageUrl = asset('celebration/' . $imageName);

                        // Only add if not already present
                        if (!in_array($imageUrl, $finalImages)) {
                            $finalImages[] = $imageUrl;
                        }
                    }
                }
            }

            // Store unique images
            $celebration->N_Image = json_encode(array_unique($finalImages), JSON_UNESCAPED_SLASHES);

            // Update other fields
            $celebration->N_Title       = $request->N_Title ?? $celebration->N_Title;
            $celebration->N_Category    = $request->N_Category ?? $celebration->N_Category;
            $celebration->N_MaleName    = $request->N_MaleName ?? $celebration->N_MaleName;
            $celebration->N_WomanName   = $request->N_WomanName ?? $celebration->N_WomanName;
            $celebration->N_Date        = $request->N_Date ?? $celebration->N_Date;
            $celebration->N_Age         = $request->N_Age ?? $celebration->N_Age;
            $celebration->N_Years       = $request->N_Years ?? $celebration->N_Years;
            $celebration->N_Mobile      = $request->N_Mobile ?? $celebration->N_Mobile;
            $celebration->N_Email       = $request->N_Email ?? $celebration->N_Email;
            $celebration->N_Address     = $request->N_Address ?? $celebration->N_Address;
            $celebration->N_Description = $request->N_Description ?? $celebration->N_Description;
            $celebration->U_Id          = $request->U_Id ?? $celebration->U_Id;
            $celebration->N_Status      = $request->N_Status ?? $celebration->N_Status;

            $celebration->save();

            // Decode before sending
            $celebration->N_Image = json_decode($celebration->N_Image, true);

            return response()->json([
                'status' => true,
                'message' => 'Celebration updated successfully',
                'data' => $celebration
            ], 200);

        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Delete a specific image from celebration by index in URL
     */
    public function deleteImage($id, $imageIndex)
    {
        try {
            // Validate image_index
            if (!is_numeric($imageIndex) || $imageIndex < 0) {
                return response()->json(['status' => false, 'message' => 'Invalid image index'], 422);
            }

            $celebration = Celebration::find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            // Get existing images
            $existingImages = is_string($celebration->N_Image) ? json_decode($celebration->N_Image, true) : [];
            $existingImages = is_array($existingImages) ? $existingImages : [];

            // Check if index is valid
            if (!isset($existingImages[$imageIndex])) {
                return response()->json(['status' => false, 'message' => 'Image index not found'], 404);
            }

            // Get the image URL to delete
            $imageToDelete = $existingImages[$imageIndex];

            // Remove image from array
            unset($existingImages[$imageIndex]);

            // Convert back to indexed array
            $updatedImages = array_values($existingImages);

            // Delete physical file if it exists
            $filePath = public_path(str_replace(asset(''), '', $imageToDelete));
            if (file_exists($filePath)) {
                unlink($filePath);
            }

            // Update celebration with new image array
            $celebration->N_Image = json_encode($updatedImages, JSON_UNESCAPED_SLASHES);
            $celebration->save();

            // Decode before sending
            $celebration->N_Image = json_decode($celebration->N_Image, true);

            return response()->json([
                'status' => true,
                'message' => 'Image deleted successfully',
                'data' => $celebration
            ], 200);

        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }


    public function destroy($id)
    {
        try {
            // Find celebration by ID
            $celebration = Celebration::find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            $celebration->delete();

            return response()->json(['status' => true, 'message' => 'Celebration deleted successfully'], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }
}
