<?php

namespace App\Http\Controllers;

use App\Models\Condolences;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CondolencesController extends Controller
{
    public function index()
    {
        try {
            // Get all celebrations with the related user data (eager loading)
            $celebrations = Condolences::with('user')->get();

            // Loop through each celebration and decode the N_Image field to remove any backslashes
            foreach ($celebrations as $celebration) {
                if (is_string($celebration->N_Image)) {
                    $celebration->N_Image = json_decode($celebration->N_Image, true);
                }
            }

            return response()->json(['status' => true, 'message' => 'Celebrations fetched successfully', 'data' => $celebrations], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'N_Title' => 'required|string',
                'U_Id' => 'required|exists:user_masters,U_Id',
                'N_Image' => 'nullable|array',
                'N_Image.*' => 'nullable|image|mimes:jpeg,jpg,png,gif,bmp,webp|max:20480',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
            }

            $user = UserMaster::where('U_Id', $request->U_Id)->first();

            // Determine N_Status
            $status = 'pending'; // Default for Role_Id = 3
            if ($user && $user->Role_Id == 2) {
                $status = 'approved'; // Direct post for Role_Id = 2
            }

            // Handle multiple image uploads
            $photoPaths = [];
            if ($request->hasFile('N_Image')) {
                foreach ($request->file('N_Image') as $image) {
                    if ($image->isValid()) {
                        $filename = time() . '_' . $image->getClientOriginalName();
                        $image->move(public_path('condolences/'), $filename);
                        $photoPaths[] = asset('condolences/' . $filename);
                    }
                }
            }

            $celebration = Condolences::create([
                'N_Title'       => $request->N_Title,
                'N_Category'    => $request->N_Category,
                'N_MaleName'    => $request->N_MaleName,
                'N_WomanName'   => $request->N_WomanName,
                'N_Date'        => $request->N_Date,
                'N_Age'         => $request->N_Age,
                'N_Years'       => $request->N_Years,
                'N_Mobile'      => $request->N_Mobile,
                'N_Email'       => $request->N_Email,
                'N_Address'     => $request->N_Address,
                'N_Description' => $request->N_Description,
                'N_Image'       => $photoPaths,
                'U_Id'          => $request->U_Id,
                'N_Status'      => $status
            ]);

            return response()->json(['status' => true, 'message' => 'Celebration created successfully', 'data' => $celebration], 201);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'N_Title'     => 'nullable|string',
                'U_Id'        => 'nullable|exists:user_masters,U_Id',
                'N_Image'     => 'sometimes|array',
                'N_Image.*'   => 'sometimes|string',
            ]);

            if ($validator->fails()) {
                return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
            }

            $celebration = Condolences::find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            // Prevent status change if already rejected
            if ($celebration->N_Status === 'rejected' && $request->has('N_Status')) {
                return response()->json(['status' => false, 'message' => 'Status cannot be changed after being rejected'], 400);
            }

            // Delete old images if new images are provided
            $oldPhotos = is_string($celebration->N_Image) ? json_decode($celebration->N_Image, true) : [];
            if ($request->has('N_Image') && is_array($oldPhotos)) {
                foreach ($oldPhotos as $oldPhoto) {
                    $filePath = public_path(parse_url($oldPhoto, PHP_URL_PATH));
                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }
                }
            }

            // Handle new base64 images
            $newPhotos = [];
            if ($request->has('N_Image') && is_array($request->N_Image)) {
                foreach ($request->N_Image as $imageData) {
                    if (strpos($imageData, 'data:image') === 0) {
                        list($type, $imageData) = explode(';', $imageData);
                        list(, $imageData) = explode(',', $imageData);
                    }

                    $directory = public_path('condolences/');
                    if (!file_exists($directory)) {
                        mkdir($directory, 0777, true);
                    }

                    $imageName = time() . '_' . uniqid() . '.png';
                    file_put_contents($directory . '/' . $imageName, base64_decode($imageData));
                    $newPhotos[] = asset('condolences/' . $imageName);
                }

                $celebration->N_Image = json_encode($newPhotos, JSON_UNESCAPED_SLASHES);
            }

            // Update fields if provided
            $celebration->N_Title       = $request->N_Title ?? $celebration->N_Title;
            $celebration->N_Category    = $request->N_Category ?? $celebration->N_Category;
            $celebration->N_MaleName    = $request->N_MaleName ?? $celebration->N_MaleName;
            $celebration->N_WomanName   = $request->N_WomanName ?? $celebration->N_WomanName;
            $celebration->N_Date        = $request->N_Date ?? $celebration->N_Date;
            $celebration->N_Age         = $request->N_Age ?? $celebration->N_Age;
            $celebration->N_Years       = $request->N_Years ?? $celebration->N_Years;
            $celebration->N_Mobile      = $request->N_Mobile ?? $celebration->N_Mobile;
            $celebration->N_Email       = $request->N_Email ?? $celebration->N_Email;
            $celebration->N_Address     = $request->N_Address ?? $celebration->N_Address;
            $celebration->N_Description = $request->N_Description ?? $celebration->N_Description;
            $celebration->U_Id          = $request->U_Id ?? $celebration->U_Id;
            $celebration->N_Status      = $request->N_Status ?? $celebration->N_Status;

            $celebration->save();

            // Decode image JSON before response (only if it's a string)
            if (is_string($celebration->N_Image)) {
                $celebration->N_Image = json_decode($celebration->N_Image, true);
            }

            return response()->json([
                'status' => true,
                'message' => 'Celebration updated successfully',
                'data' => $celebration
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        try {
            // Find celebration by ID with the related user data (eager loading)
            $celebration = Condolences::with('user')->find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            // Decode the N_Image field to remove any backslashes
            if (is_string($celebration->N_Image)) {
                $celebration->N_Image = json_decode($celebration->N_Image, true);
            }

            return response()->json(['status' => true, 'message' => 'Celebration fetched successfully', 'data' => $celebration], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        try {
            // Find celebration by ID
            $celebration = Condolences::find($id);

            if (!$celebration) {
                return response()->json(['status' => false, 'message' => 'Celebration not found'], 404);
            }

            $celebration->delete();

            return response()->json(['status' => true, 'message' => 'Celebration deleted successfully'], 200);
        } catch (\Exception $e) {
            return response()->json(['status' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }
}
