<?php

namespace App\Http\Controllers;

use App\Models\DiscountMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class DiscountMasterController extends Controller
{
    public function index()
    {
        $discounts = DiscountMaster::with(['discountType', 'discountValueType'])->get();
        return response()->json($discounts, 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'D_Name' => 'required|string|max:255',
            'St_Date' => 'required|date',
            'En_Date' => 'required|date|after_or_equal:St_Date',
            'DT_Id' => 'required|integer|exists:discount_types,DT_Id',
            'DVT_Id' => 'required|integer|exists:dis_val_types,DVT_Id',
            'Dic_Value' => 'required|string|max:100',
            'quentity' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $discount = DiscountMaster::create($request->only([
                'D_Name',
                'St_Date',
                'En_Date',
                'DT_Id',
                'DVT_Id',
                'Dic_Value',
                'quentity',
            ]));
            return response()->json(['message' => 'Discount created successfully.', 'data' => $discount], 201);
        } catch (\Exception $e) {
            Log::error('Discount creation failed: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Failed to create discount.', 'details' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        $discount = DiscountMaster::with(['discountType', 'discountValueType'])->find($id);
        if (!$discount) {
            return response()->json(['error' => 'Discount not found.'], 404);
        }
        return response()->json($discount, 200);
    }

    public function update(Request $request, $id)
    {
        $discount = DiscountMaster::find($id);
        if (!$discount) {
            return response()->json(['error' => 'Discount not found.'], 404);
        }

        $validator = Validator::make($request->all(), [
            'D_Name' => 'sometimes|string|max:255',
            'St_Date' => 'sometimes|date',
            'En_Date' => 'sometimes|date|after_or_equal:St_Date',
            'DT_Id' => 'sometimes|integer|exists:discount_types,DT_Id',
            'DVT_Id' => 'sometimes|integer|exists:dis_val_types,DVT_Id',
            'Dic_Value' => 'sometimes|string|max:100',
            'quentity' => 'sometimes|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $discount->update($request->only([
                'D_Name',
                'St_Date',
                'En_Date',
                'DT_Id',
                'DVT_Id',
                'Dic_Value',
                'quentity',
            ]));
            return response()->json(['message' => 'Discount updated successfully.', 'data' => $discount], 200);
        } catch (\Exception $e) {
            Log::error('Discount update failed: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Update failed.', 'details' => $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        $discount = DiscountMaster::find($id);
        if (!$discount) {
            return response()->json(['error' => 'Discount not found.'], 404);
        }

        try {
            $discount->delete();
            return response()->json(['message' => 'Discount deleted successfully.'], 200);
        } catch (\Exception $e) {
            Log::error('Discount deletion failed: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Delete failed.', 'details' => $e->getMessage()], 500);
        }
    }
}
