<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Donate;

use Exception;
use Illuminate\Support\Facades\Validator;

class DonateController extends Controller
{
    // Create a donation record
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'D_amount' => 'required|numeric',  // minimum 5 rupees
                'U_ID'     => 'required|exists:user_masters,U_Id',
                'D_id'     => 'required|exists:donations,D_id',
                'Date'     => 'required|date'
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $donate = Donate::create($validator->validated());
            return response()->json(['message' => 'Donation recorded successfully', 'data' => $donate], 201);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to record donation', 'details' => $e->getMessage()], 500);
        }
    }

    // List all donations
    public function index()
    {
        try {
            $donations = Donate::with(['user', 'donation'])->get();
            return response()->json(['data' => $donations]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to fetch donations', 'details' => $e->getMessage()], 500);
        }
    }

    // Show single donation record
    public function show($id)
    {
        try {
            $donate = Donate::with(['user', 'donation'])->findOrFail($id);
            return response()->json(['data' => $donate]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Donation record not found', 'details' => $e->getMessage()], 404);
        }
    }

    // Update donation record
    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'D_amount' => 'required|numeric',
                'U_ID'     => 'required|exists:user_masters,U_Id',
                'D_id'     => 'required|exists:donations,D_id',
                'Date'     => 'required|date'
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $donate = Donate::findOrFail($id);
            $donate->update($validator->validated());

            return response()->json(['message' => 'Donation record updated successfully', 'data' => $donate]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update donation record', 'details' => $e->getMessage()], 500);
        }
    }

    // Delete donation record
    public function destroy($id)
    {
        try {
            $donate = Donate::findOrFail($id);
            $donate->delete();

            return response()->json(['message' => 'Donation record deleted successfully']);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to delete donation record', 'details' => $e->getMessage()], 500);
        }
    }
}
