<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Donation;
use Exception;
use Illuminate\Support\Facades\Validator;

class DonationController extends Controller
{
    // Create Donation
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'D_name'       => 'required|string',
                'D_des'        => 'nullable|string',
                'goal_amount'  => 'required|numeric|min:1',
                'mini_amount'  => 'required|numeric',
                'Start_date'   => 'required|date',
                'End_date'     => 'required|date|after_or_equal:Start_date',
                'U_ID'         => 'required|exists:user_masters,U_Id',
                'Cat_id'         => 'required|exists:categories,Cat_id',
                'status'       => 'in:0,1',
                'Pop_up'       => 'in:Y,N'
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $donation = Donation::create([
                ...$validator->validated(),
                'Pop_up' => 'N',
                'status' => 1
            ]);

            // eager load user relationship
            $donation->load('user');

            return response()->json(['message' => 'Donation created successfully', 'data' => $donation], 201);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to create donation', 'details' => $e->getMessage()], 500);
        }
    }

    // List All Donations
    public function index()
    {
        try {
            $donations = Donation::with('user', 'category')->get(); // eager load users
            return response()->json(['data' => $donations]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to fetch donations', 'details' => $e->getMessage()], 500);
        }
    }

    // Show Single Donation by ID
    public function show($id)
    {
        try {
            $donation = Donation::with('user')->findOrFail($id); // eager load user
            return response()->json(['data' => $donation]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Donation not found', 'details' => $e->getMessage()], 404);
        }
    }

    // Update Donation
public function update(Request $request, $id)
{
    try {
        $validator = Validator::make($request->all(), [
            'D_name'       => 'required|string',
            'D_des'        => 'nullable|string',
            'goal_amount'  => 'required|numeric|min:1',
            'mini_amount'  => 'required|numeric',
            'Start_date'   => 'required|date',
            'End_date'     => 'required|date|after_or_equal:Start_date',
            'U_ID'         => 'required|exists:user_masters,U_Id',
            'Cat_id'         =>'required|exists:categories,Cat_id',
            'status'       => 'nullable|in:0,1' // optional, only if user provides
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $donation = Donation::findOrFail($id);

        $data = $validator->validated();

        // If status is not provided, retain old status
        if (!isset($data['status'])) {
            unset($data['status']); // don't override existing status
        }

        $donation->update($data);

        // Eager load user relation
        $donation->load('user');

        return response()->json([
            'message' => 'Donation updated successfully',
            'data' => $donation
        ], 200);
    } catch (Exception $e) {
        return response()->json([
            'error' => 'Failed to update donation',
            'details' => $e->getMessage()
        ], 500);
    }
}



    // Delete Donation
    public function destroy($id)
    {
        try {
            $donation = Donation::findOrFail($id);
            $donation->delete();

            return response()->json(['message' => 'Donation deleted successfully']);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to delete donation', 'details' => $e->getMessage()], 500);
        }
    }

    // Trigger Pop-Up
    public function triggerPopUp($id)
    {
        try {
            $donation = Donation::findOrFail($id);
            $donation->Pop_up = 'Y';
            $donation->save();

            $donation->load('user');

            return response()->json(['message' => 'Popup set to Y', 'data' => $donation]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to update popup', 'details' => $e->getMessage()], 500);
        }
    }

    // Activate Donation
    public function activate($id)
    {
        try {
            $donation = Donation::findOrFail($id);
            $donation->status = 1;
            $donation->save();

            $donation->load('user');

            return response()->json(['message' => 'Donation activated', 'data' => $donation]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to activate', 'details' => $e->getMessage()], 500);
        }
    }

    // Deactivate Donation
    public function deactivate($id)
    {
        try {
            $donation = Donation::findOrFail($id);
            $donation->status = 0;
            $donation->save();

            $donation->load('user');

            return response()->json(['message' => 'Donation deactivated', 'data' => $donation]);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to deactivate', 'details' => $e->getMessage()], 500);
        }
    }
}
