<?php

namespace App\Http\Controllers;

use App\Models\Gallary;
use App\Models\Gallery;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class GalleryController extends Controller
{
    // 🔹 STORE Gallery Entry (Role_Id = 2)
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'CE_Name' => 'required|string|max:255',
                'CE_Date' => 'nullable|date',
                'CE_Place' => 'nullable|string|max:255',
                'CE_Description' => 'nullable|string',
                'CE_Photo' => 'required|array|min:1', // Changed 'required' to 'nullable'
                'CE_Photo.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:20480',
                'U_Id' => 'required|exists:user_masters,U_Id',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            $user = UserMaster::find($request->U_Id);
            if (!$user || $user->Role_Id != 2) {
                return response()->json(['error' => 'Only Role_Id = 2 users can post galleries.'], 403);
            }

            // ✅ Handle multiple image uploads
            $photoPaths = [];
            if ($request->hasFile('CE_Photo')) {
                foreach ($request->file('CE_Photo') as $image) {
                    if ($image->isValid()) {
                        $filename = time() . '_' . $image->getClientOriginalName();
                        $image->move(public_path('gallary/photos'), $filename);
                        $photoPaths[] = asset('gallary/photos/' . $filename);
                    }
                }
            }

            $gallery = Gallary::create([
                'CE_Name' => $request->CE_Name,
                'CE_Date' => $request->CE_Date,
                'CE_Place' => $request->CE_Place,
                'CE_Description' => $request->CE_Description,
                'CE_Photo' => $photoPaths,
                'U_Id' => $request->U_Id,
            ]);

            return response()->json(['data' => $gallery, 'message' => 'Gallery created successfully'], 201);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Server error', 'message' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $gallery = Gallary::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'CE_Name' => 'nullable|string|max:255',
                'CE_Date' => 'nullable|date',
                'CE_Place' => 'nullable|string|max:255',
                'CE_Description' => 'nullable|string',
                'CE_Photo' => 'nullable|array',
                'CE_Photo.*' => 'nullable|string',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            // Ensure CE_Photo is a string before decoding
            $oldPhotos = is_string($gallery->CE_Photo) ? json_decode($gallery->CE_Photo, true) : [];

            // Initialize new photos array
            $newPhotos = $oldPhotos; // Preserve old photos by default

            // Handle new image uploads (Base64) only if CE_Photo is provided
            if ($request->has('CE_Photo') && is_array($request->CE_Photo)) {
                // Remove old images only if new images are provided
                if (is_array($oldPhotos)) {
                    foreach ($oldPhotos as $oldPhoto) {
                        $filePath = public_path(parse_url($oldPhoto, PHP_URL_PATH));
                        if (file_exists($filePath)) {
                            unlink($filePath); // Delete old image file
                        }
                    }
                }

                $newPhotos = []; // Reset new photos array
                foreach ($request->CE_Photo as $imageData) {
                    // Validate Base64 string
                    if (!empty($imageData) && strpos($imageData, 'data:image') === 0) {
                        list($type, $imageData) = explode(';', $imageData);
                        list(, $imageData) = explode(',', $imageData);

                        $directory = public_path('gallary/photos');
                        if (!file_exists($directory)) {
                            mkdir($directory, 0777, true);
                        }

                        $imageName = time() . '_' . uniqid() . '.png';
                        file_put_contents($directory . '/' . $imageName, base64_decode($imageData));

                        $newPhotos[] = asset('gallary/photos/' . $imageName);
                    }
                }
            }

            // Store new photos
            $gallery->CE_Photo = $newPhotos; // Will be casted as JSON because of `$casts` in the model

            // Update other fields
            $gallery->fill($request->except(['CE_Photo']));
            $gallery->save();

            return response()->json([
                'data' => $gallery->toArray(),
                'message' => 'Gallery updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to update gallery',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function index()
    {
        try {
            $entries = Gallary::with('user')->get();

            return response()->json([
                'status' => 'success',
                'data' => $entries
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $entry = Gallary::with('user')->find($id);

            if (!$entry) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Gallery entry not found.'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => $entry
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $entry = Gallary::find($id);

            if (!$entry) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Gallery entry not found.'
                ], 404);
            }

            $entry->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Gallery entry deleted successfully.'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
