<?php

namespace App\Http\Controllers;

use App\Models\JobApplication;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class JobApplicationController extends Controller
{
    public function index()
    {
        return response()->json([
            'status' => true,
            'data' => JobApplication::with('user', 'job')->get()
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Name' => 'required|string',
            'Mobile' => 'required|string',
            'U_Id' => 'required|exists:user_masters,U_Id',
            'Job_Id' => 'required|exists:jobs,id',
            'Resume' => 'required|file|mimes:jpg,jpeg,pdf,doc,docx',
            'Description' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'message' => $validator->errors()], 422);
        }

        $user = UserMaster::where('U_Id', $request->U_Id)->first();

        if (!$user || $user->Role_Id != 3) {
            return response()->json([
                'status' => false,
                'message' => 'Only community members (Role_Id = 3) can apply for jobs.'
            ], 403);
        }
        // ✅ Store Resume File (Ensure "public/Resume" folder exists)
        $resumePath = null;
        if ($request->hasFile('Resume')) {
            $file = $request->file('Resume');
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path('Resume'), $filename);
            $resumePath = asset('Resume/' . $filename); // Public URL
        }

        $application = JobApplication::create([
            'Name' => $request->Name,
            'Mobile' => $request->Mobile,
            'U_Id' => $request->U_Id,
            'Comm_Id' => $request->Comm_Id,
            'Job_Id' => $request->Job_Id,
            'Resume' => $resumePath,
            'status' => $request->status ?? 'pending', // Default to pending if not provided
            'Description' => $request->Description
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Job application submitted.',
            'data' => $application
        ]);
    }

    public function show($id)
    {
        $application = JobApplication::with('user', 'job')->find($id);

        if (!$application) {
            return response()->json(['status' => false, 'message' => 'Application not found.'], 404);
        }

        return response()->json(['status' => true, 'data' => $application]);
    }

    public function update(Request $request, $id)
    {
        $application = JobApplication::find($id);

        if (!$application) {
            return response()->json(['status' => false, 'message' => 'Application not found.'], 404);
        }
        // 🔹 Prevent Rejected -> Accepted/Pending
        if ($application->status === 'rejected' && $request->status !== 'rejected') {
            return response()->json(['error' => 'Rejected applications cannot be changed'], 403);
        }

        $validator = Validator::make($request->all(), [
            'Name' => 'sometimes|string',
            'Mobile' => 'sometimes|digits:10',
            'U_Id' => 'sometimes|exists:user_masters,U_Id',
            'Job_Id' => 'sometimes|exists:jobs,id',
            'Resume' => 'sometimes|string', // Base64 validation
            'status' => 'sometimes|in:pending,accepted,rejected', // Status validation
            'Description' => 'sometimes|string' // ✅ Allow updating Description
        ]);
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        if ($request->Resume) {
            $base64String = $request->Resume;
            $fileName = time() . '_resume.pdf';
            $filePath = public_path('Resume/') . $fileName;

            file_put_contents($filePath, base64_decode($base64String));

            $application->Resume = asset('Resume/' . $fileName);
        }


        $application->update($request->only(['Name', 'Mobile', 'U_Id', 'Job_Id', 'status', 'Description']));

        return response()->json([
            'status' => true,
            'message' => 'Application updated.',
            'data' => $application
        ]);
    }

    public function destroy($id)
    {
        $application = JobApplication::find($id);

        if (!$application) {
            return response()->json(['status' => false, 'message' => 'Application not found.'], 404);
        }

        $application->delete();

        return response()->json(['status' => true, 'message' => 'Application deleted.']);
    }
}
