<?php

namespace App\Http\Controllers;

use App\Models\Job;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class JobController extends Controller
{
    public function index()
    {

        $jobs = Job::with('user')->get();

        return response()->json([
            'status' => 'success',
            'message' => 'Jobs retrieved successfully',
            'data' => $jobs
        ], 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Company_Name' => 'required|string',
            'Job_Title' => 'required|string',
            'Job_Description' => 'nullable|string',
            'website' => 'nullable|url',
            'U_Id' => 'required|integer|exists:user_masters,U_Id', // Ensure U_Id exists in user_masters
            'Location' => 'nullable|string',
            'City' => 'nullable|string',
            'State' => 'nullable|string',
            'Status' => 'in:open,close',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // // Check if the user with U_Id has Role_Id = 2 in user_masters
        // $user = DB::table('user_masters')->where('U_Id', $request->U_Id)->first();

        // if (!$user || $user->Role_Id !== 2) {
        //     return response()->json(['error' => 'Unauthorized. Only users with Role_Id = 2 can create jobs.'], 403);
        // }

        $job = Job::create([
            'Company_Name' => $request->Company_Name,
            'Job_Title' => $request->Job_Title,
            'Job_Description' => $request->Job_Description,
            'website' => $request->website,
            'U_Id' => $request->U_Id,
            'Status' => $request->Status ?? 'open',
            'Location' => $request->Location,
            'City' => $request->City,
            'State' => $request->State,
        ]);

        return response()->json($job, 201);
    }

    public function show($id)
    {
        $job = Job::find($id);

        if (!$job) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job not found'
            ], 404);
        }

        return response()->json($job, 200);
    }

    public function update(Request $request, $id)
    {
        $job = Job::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'U_Id' => 'sometimes|integer',
            'Company_Name' => 'sometimes|string',
            'Job_Title' => 'sometimes|string',
            'Job_Description' => 'sometimes|nullable|string',
            'website' => 'sometimes|nullable|url',
            'Location' => 'sometimes|nullable|string',
            'City' => 'sometimes|nullable|string',
            'State' => 'sometimes|nullable|string',
            'Status' => 'sometimes|in:open,close',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $job->update($request->only([
            'Company_Name',
            'Job_Title',
            'Job_Description',
            'website',
            'Location',
            'City',
            'State',
            'Status',
        ]));

        return response()->json([
            'status' => 'success',
            'message' => 'Job updated successfully',
            'data' => $job
        ], 200);
    }

    public function destroy($id)
    {
        $job = Job::find($id);

        if (!$job) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job not found'
            ], 404);
        }

        $job->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Job deleted successfully'
        ], 200);
    }
}
