<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\UserMaster;
use App\Models\Login;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class LoginController extends Controller
{
    public function login(Request $request)
    {
        // Validate incoming request
        $validator = Validator::make($request->all(), [
            'U_Email'   => 'required_without:U_Mobile|email',
            'U_Mobile'  => 'required_without:U_Email|string',
            'U_Password' => 'required|string|min:8',
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Choose the login ID (email or mobile)
        $loginId = $request->U_Email ?? $request->U_Mobile;
        $password = $request->U_Password;

        // Search user by email or mobile
        $user = UserMaster::where('U_Email', $loginId)
            ->orWhere('U_Mobile', $loginId)
            ->first();

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not found.',
            ], 404);
        }

        // Check if the user's status is 1 (active)
        if ($user->U_Status != 1) {
            return response()->json([
                'status' => 'error',
                'message' => 'Your account is inactive. Please contact support.',
            ], 403);
        }

        // Check if password is correct
        if (!Hash::check($password, $user->U_Password)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid credentials.',
            ], 401);
        }

        // Retrieve Role data
        $role = \App\Models\RoleMaster::find($user->Role_Id);

        // Store login event in the logins table (without storing the password)
        $loginData = new Login();
        $loginData->U_Id = $user->U_Id;
        $loginData->email = $user->U_Email;
        $loginData->mobile = $user->U_Mobile;
        $loginData->save();

        return response()->json([
            'status' => 'success',
            'message' => 'Login successful.',
            'data' => [
                'user' => [
                    'U_Id'     => $user->U_Id,
                    'U_Name'   => $user->U_Name,
                    'U_Email'  => $user->U_Email,
                    'U_Mobile' => $user->U_Mobile,
                    'Comm_Id'  => $user->Comm_Id,
                    'Role_Id'  => $role->Role_Id,
                    'Role_Name' => $role->Role_Name,
                ],
            ],
        ], 200);
    }
}
