<?php

namespace App\Http\Controllers;

use App\Models\Marketplace;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class MarketplaceController extends Controller
{

    public function index()
    {
        try {
            // ✅ Eager load full user data for all marketplace items
            $marketplaceItems = Marketplace::with('user')->get();

            return response()->json([
                'message' => 'All marketplace items fetched successfully',
                'data' => $marketplaceItems
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'M_Category' => 'required|in:Electronics,Home Appliances,Furniture,Clothes,Footwear,Books,Vehicles,Real Estate,Toys,Sports,Fitness,Handicrafts,Others',
                'M_Title' => 'required|string',
                'M_Description' => 'nullable|string',
                'M_Name' => 'required|string',
                'M_Mobile' => 'required|string',
                'M_Address' => 'required|string',
                'M_Image' => 'required|array',
                'M_Image.*' => 'image|mimes:jpeg,png,jpg,gif',
                'M_Price' => 'required|numeric',
                'M_Status' => 'in:Active,Inactive',
                'U_Id' => 'required|exists:user_masters,U_Id',
            ]);


            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            // ✅ Handle multiple image uploads
            $photoPaths = [];
            if ($request->hasFile('M_Image')) {
                foreach ($request->file('M_Image') as $image) {
                    if ($image->isValid()) {
                        $filename = time() . '_' . $image->getClientOriginalName();
                        $image->move(public_path('marketplace/'), $filename);
                        $photoPaths[] = asset('marketplace/' . $filename);
                    }
                }
            }

            // ✅ Store Data
            $marketplace = Marketplace::create([
                'M_Category' => $request->M_Category,
                'M_Title' => $request->M_Title,
                'M_Description' => $request->M_Description,
                'M_Name' => $request->M_Name,
                'M_Mobile' => $request->M_Mobile,
                'M_Address' => $request->M_Address,
                'M_Image' => $photoPaths, // ✅ Store as JSON
                'M_Price' => $request->M_Price,
                'M_Status' => $request->M_Status ?? 'Active',
                'U_Id' => $request->U_Id,
            ]);

            return response()->json([
                'message' => 'Marketplace item added successfully',
                'data' => $marketplace
            ], 201);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $marketplace = Marketplace::find($id);
            if (!$marketplace) {
                return response()->json(['error' => 'Marketplace item not found'], 404);
            }

            $validator = Validator::make($request->all(), [
                'M_Category' => 'sometimes|in:Electronics,Home Appliances,Furniture,Clothes,Footwear,Books,Vehicles,Real Estate,Toys,Sports,Fitness,Handicrafts,Others',
                'M_Title' => 'sometimes|string',
                'M_Description' => 'nullable|string',
                'M_Name' => 'sometimes|string',
                'M_Mobile' => 'sometimes|string',
                'M_Address' => 'sometimes|string',
                'M_Image' => 'sometimes|array', // Base64 array
                'M_Image.*' => 'string', // Each item is base64 encoded
                'M_Image.*' => 'image|mimes:jpeg,png,jpg,gif|max:20480',
                'M_Price' => 'sometimes|numeric',
                'M_Status' => 'in:Active,Inactive',
                'U_Id' => 'sometimes|exists:user_masters,U_Id',
            ]);

            if ($validator->fails()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }

            // 🖼️ Handle base64 image updates (append to existing)
            $existingImages = $marketplace->M_Image ?? [];
            $newImagePaths = [];

            if ($request->has('M_Image')) {
                foreach ($request->M_Image as $index => $base64Image) {
                    if (preg_match('/^data:image\/(\w+);base64,/', $base64Image, $type)) {
                        $base64Image = substr($base64Image, strpos($base64Image, ',') + 1);
                        $type = strtolower($type[1]);

                        if (!in_array($type, ['jpg', 'jpeg', 'png', 'gif'])) {
                            return response()->json(['error' => "Invalid image type at index $index"], 422);
                        }

                        $base64Image = base64_decode($base64Image);
                        if ($base64Image === false) {
                            return response()->json(['error' => "Base64 decoding failed at index $index"], 422);
                        }

                        $filename = time() . '_' . uniqid() . '.' . $type;
                        $path = public_path('marketplace/' . $filename);
                        file_put_contents($path, $base64Image);

                        $newImagePaths[] = asset('marketplace/' . $filename);
                    }
                }
            }

            // 🔁 Merge old and new images (or replace if you want)
            $allImages = array_merge($existingImages, $newImagePaths);

            // 🔄 Update values
            $marketplace->update([
                'M_Category' => $request->M_Category ?? $marketplace->M_Category,
                'M_Title' => $request->M_Title ?? $marketplace->M_Title,
                'M_Description' => $request->M_Description ?? $marketplace->M_Description,
                'M_Name' => $request->M_Name ?? $marketplace->M_Name,
                'M_Mobile' => $request->M_Mobile ?? $marketplace->M_Mobile,
                'M_Address' => $request->M_Address ?? $marketplace->M_Address,
                'M_Image' => $allImages,
                'M_Price' => $request->M_Price ?? $marketplace->M_Price,
                'M_Status' => $request->M_Status ?? $marketplace->M_Status,
                'U_Id' => $request->U_Id ?? $marketplace->U_Id,
            ]);

            return response()->json([
                'message' => 'Marketplace item updated successfully',
                'data' => $marketplace
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        try {
            // ✅ Fetch Marketplace Item by ID with Full User Data
            $marketplaceItem = Marketplace::with('user')->find($id);

            if (!$marketplaceItem) {
                return response()->json(['error' => 'Marketplace item not found'], 404);
            }

            return response()->json([
                'message' => 'Marketplace item fetched successfully',
                'data' => $marketplaceItem
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $marketplace = Marketplace::find($id);

            if (!$marketplace) {
                return response()->json(['error' => 'Marketplace item not found'], 404);
            }

            $marketplace->delete(); // Soft delete

            return response()->json([
                'message' => 'Marketplace item soft deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function trashed()
    {
        try {
            $trashedItems = Marketplace::onlyTrashed()->get();

            return response()->json([
                'message' => 'Soft deleted marketplace items fetched successfully',
                'data' => $trashedItems
            ], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
