<?php

namespace App\Http\Controllers;

use App\Mail\OtpMail;
use App\Models\OtpLog;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class OtpController extends Controller
{
    public function sendOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'U_Email' => 'required|email|exists:user_masters,U_Email'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        $user = UserMaster::where('U_Email', $request->U_Email)->first();
        $otp = rand(100000, 999999);
        $expiry = now()->addMinutes(5);

        OtpLog::create([
            'U_Id' => $user->U_Id,
            'OTP_Char' => $otp,
            'OPT_Type' => 'P',
            'OPT_Status' => 'N',
            'SendingDate' => now()->toDateString(),
            'SendingTime' => now()->toTimeString(),
            'ExpieryDate' => $expiry->toDateString(),
            'ExpieryTime' => $expiry->toTimeString(),
        ]);

        Mail::to($user->U_Email)->send(new OtpMail($otp));

        return response()->json(['message' => 'OTP sent successfully']);
    }

    public function verifyOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'OTP_Char' => 'required|max:6'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        $otpLog = OtpLog::where('OTP_Char', $request->OTP_Char)
            ->where('OPT_Status', 'N')
            ->whereRaw("STR_TO_DATE(CONCAT(ExpieryDate, ' ', ExpieryTime), '%Y-%m-%d %H:%i:%s') >= ?", [now()])
            ->first();

        if (!$otpLog) {
            return response()->json(['message' => 'Invalid or expired OTP'], 400);
        }

        // Mark OTP as Verified
        $otpLog->update(['OPT_Status' => 'V']);

        // Store verified user ID in cache for 10 minutes
        Cache::put('verified_user_id', $otpLog->U_Id, now()->addMinutes(10));

        return response()->json(['message' => 'OTP verified successfully']);
    }

    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'new_password' => 'required|string|min:6',
            'confirm_password' => 'required|string|same:new_password'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        // Retrieve user ID from cache
        $userId = Cache::get('verified_user_id'); // ❌ Removed `$request->user()->id`

        if (!$userId) {
            return response()->json(['message' => 'OTP verification required'], 400);
        }

        $user = UserMaster::find($userId);
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $user->U_Password = Hash::make($request->new_password);
        $user->save();

        // Clear cache after password reset
        Cache::forget('verified_user_id');

        return response()->json(['message' => 'Password reset successfully']);
    }
}
