<?php

namespace App\Http\Controllers;

use App\Models\DiscountMaster;
use App\Models\PackageMaster;
use App\Models\PriceTypeValue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PackageMasterController extends Controller
{

    public function store(Request $request)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'PackageName' => 'required|string',
            'MaxUsers' => 'required|integer',
            'Duration_Value' => 'required|integer',
            'DT_Id' => 'required|exists:duration_types,DT_Id',
            'D_Id' => 'required|array',
            'D_Id.*' => 'exists:discount_masters,D_Id',
            'Dis_Merge' => 'required|integer|in:' . implode(',', $request->D_Id),
            'PVT_Id' => 'required|exists:price_type_values,PVT_Id',
            'Fea_Id' => 'required|array',
            'Fea_Id.*' => 'exists:package_features,Package_Fea_Id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Get the base price value from PriceTypeValue using PVT_Id
            $priceValue = PriceTypeValue::where('PVT_Id', $request->PVT_Id)->value('Price_Value');
            $finalPrice = (float) $priceValue;  // Initialize Final_Price with base price

            // Get the discount that matches the Dis_Merge ID
            $discount = DiscountMaster::with('discountType', 'discountValueType')
                ->where('D_Id', $request->Dis_Merge)
                ->first();

            if ($discount) {
                $discountValue = (float) $discount->Dic_Value;
                $discountType = strtolower($discount->discountValueType->VT_Name); // Either 'flat discount' or 'percentage discount'

                // Ensure discountType is correctly matched, considering exact case from DB
                if (strcasecmp($discountType, 'percentage discount') === 0) {
                    // Apply percentage discount (reduce by percentage)
                    $finalPrice -= ($priceValue * $discountValue / 100);
                } elseif (strcasecmp($discountType, 'flat discount') === 0) {
                    // Apply flat discount (subtract flat amount)
                    $finalPrice -= $discountValue;
                }
            }

            // Ensure the Final_Price doesn't go below zero
            $finalPrice = max(0, $finalPrice);

            // Get duration type and calculate total duration
            $durationType = DB::table('duration_types')->where('DT_Id', $request->DT_Id)->value('Dur_Type');
            $durationFactor = $this->getDurationFactor($durationType);
            $totalDuration = $request->Duration_Value * $durationFactor;

            // Create the package with the calculated Final_Price
            $package = PackageMaster::create([
                'PackageName' => $request->PackageName,
                'Description' => $request->Description,
                'Status' => $request->Status ?? 'A',
                'MaxUsers' => $request->MaxUsers,
                'Duration_Value' => $request->Duration_Value,
                'DT_Id' => $request->DT_Id,
                'Dis_Merge' => $request->Dis_Merge,
                'PVT_Id' => $request->PVT_Id,
                'Final_Price' => $finalPrice,  // Dynamically calculated
                'Total_Duration' => $totalDuration
            ]);

            // Sync features (many-to-many relationship)
            $package->features()->sync($request->Fea_Id);

            // Sync all selected discounts
            $package->discounts()->sync($request->D_Id);

            // Fetch Fea_Id and D_Id arrays
            $Fea_Ids = $package->features()->pluck('package_features.Package_Fea_Id')->toArray();
            $D_Ids = $package->discounts()->pluck('discount_masters.D_Id')->toArray();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Package created successfully!',
                'data' => [
                    'PackageId' => $package->PackageId,
                    'PackageName' => $package->PackageName,
                    'Description' => $package->Description,
                    'Status' => $package->Status,
                    'MaxUsers' => $package->MaxUsers,
                    'Duration_Value' => $package->Duration_Value,
                    'DT_Id' => $package->DT_Id,
                    'Dis_Merge' => $package->Dis_Merge,
                    'PVT_Id' => $package->PVT_Id,
                    'Final_Price' => $package->Final_Price,  // Dynamically calculated
                    'Total_Duration' => $package->Total_Duration,
                    'created_at' => $package->created_at,
                    'updated_at' => $package->updated_at,
                    'Fea_Id' => $Fea_Ids,
                    'D_Id' => $D_Ids
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    public function update(Request $request, $id)
    {
        $package = PackageMaster::find($id);

        if (!$package) {
            return response()->json([
                'status' => false,
                'message' => 'Package not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'PackageName' => 'sometimes|required|string',
            'Description' => 'nullable|string',
            'Status' => 'nullable|string|in:A,D',
            'MaxUsers' => 'sometimes|required|integer',
            'Duration_Value' => 'sometimes|required|integer',
            'DT_Id' => 'sometimes|required|exists:duration_types,DT_Id',
            'D_Id' => 'sometimes|required|array',
            'D_Id.*' => 'exists:discount_masters,D_Id',
            'Dis_Merge' => 'sometimes|required|integer',
            'PVT_Id' => 'sometimes|required|exists:price_type_values,PVT_Id',
            'Fea_Id' => 'sometimes|required|array',
            'Fea_Id.*' => 'exists:package_features,Package_Fea_Id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Update fields if present
            if ($request->has('PackageName')) $package->PackageName = $request->PackageName;
            if ($request->has('Description')) $package->Description = $request->Description;
            if ($request->has('Status')) $package->Status = $request->Status;
            if ($request->has('MaxUsers')) $package->MaxUsers = $request->MaxUsers;
            if ($request->has('Duration_Value')) $package->Duration_Value = $request->Duration_Value;
            if ($request->has('DT_Id')) $package->DT_Id = $request->DT_Id;
            if ($request->has('Dis_Merge')) $package->Dis_Merge = $request->Dis_Merge;
            if ($request->has('PVT_Id')) $package->PVT_Id = $request->PVT_Id;

            // Recalculate Final_Price if PVT_Id or Dis_Merge updated
            if ($request->has('PVT_Id') || $request->has('Dis_Merge')) {
                $priceValue = PriceTypeValue::where('PVT_Id', $package->PVT_Id)->value('Price_Value');
                $finalPrice = (float) $priceValue;

                $discount = DiscountMaster::with('discountType', 'discountValueType')
                    ->where('D_Id', $package->Dis_Merge)
                    ->first();

                if ($discount) {
                    $discountValue = (float) $discount->Dic_Value;
                    $discountType = strtolower($discount->discountValueType->VT_Name);

                    if (strcasecmp($discountType, 'percentage discount') === 0) {
                        $finalPrice -= ($priceValue * $discountValue / 100);
                    } elseif (strcasecmp($discountType, 'flat discount') === 0) {
                        $finalPrice -= $discountValue;
                    }
                }

                $package->Final_Price = max(0, $finalPrice);
            }

            // Calculate Total_Duration if Duration_Value or DT_Id is present
            if ($request->has('Duration_Value') || $request->has('DT_Id')) {
                $durationType = DB::table('duration_types')->where('DT_Id', $package->DT_Id)->value('Dur_Type');
                $durationFactor = $this->getDurationFactor($durationType);
                $package->Total_Duration = $package->Duration_Value * $durationFactor;
            }

            $package->save();

            // Sync features if sent
            if ($request->has('Fea_Id')) {
                $package->features()->sync($request->Fea_Id);
            }

            // Sync discounts if sent
            if ($request->has('D_Id')) {
                $package->discounts()->sync($request->D_Id);
            }

            // Get updated IDs
            $Fea_Ids = $package->features()->pluck('package_features.Package_Fea_Id')->toArray();
            $D_Ids = $package->discounts()->pluck('discount_masters.D_Id')->toArray();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Package updated successfully!',
                'data' => [
                    'PackageId' => $package->PackageId,
                    'PackageName' => $package->PackageName,
                    'Description' => $package->Description,
                    'Status' => $package->Status,
                    'MaxUsers' => $package->MaxUsers,
                    'Duration_Value' => $package->Duration_Value,
                    'DT_Id' => $package->DT_Id,
                    'Dis_Merge' => $package->Dis_Merge,
                    'PVT_Id' => $package->PVT_Id,
                    'Final_Price' => $package->Final_Price,
                    'Total_Duration' => $package->Total_Duration,
                    'updated_at' => $package->updated_at,
                    'Fea_Id' => $Fea_Ids,
                    'D_Id' => $D_Ids
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        // Fetch the package by its ID with eager loading for the related models: features, durationType, discounts, and priceType
        $package = PackageMaster::with([
            'features',      // Eager load the features relationship
            'durationType',  // Eager load the duration type relationship
            'discounts',     // Eager load the discounts relationship
            'priceType'      // Eager load the price type relationship
        ])->find($id);

        // Check if the package exists
        if (!$package) {
            return response()->json([
                'status' => false,
                'message' => 'Package not found.'
            ], 404);
        }

        // Map the result to include only Fea_Id, D_Id and related data
        $packageData = [
            'PackageId'      => $package->PackageId,
            'PackageName'    => $package->PackageName,
            'Description'    => $package->Description,
            'Status'         => $package->Status,
            'MaxUsers'       => $package->MaxUsers,
            'Duration_Value' => $package->Duration_Value,
            'DT_Id'          => $package->DT_Id,
            'Dis_Merge'      => $package->Dis_Merge,
            'PVT_Id'         => $package->PVT_Id,
            'Final_Price'    => $package->Final_Price,
            'Total_Duration' => $package->Total_Duration,
            'Fea_Id'         => $package->features->pluck('Package_Fea_Id')->toArray(), // Get Fea_Id as an array
            'D_Id'           => $package->discounts->pluck('D_Id')->toArray(), // Get D_Id as an array
            'created_at'     => $package->created_at,
            'updated_at'     => $package->updated_at,
            'features'       => $package->features,      // Include features data
            'durationType'   => $package->durationType,  // Include duration type data
            'discounts'      => $package->discounts,     // Include discounts data
            'priceType'      => $package->priceType,     // Include price type data
        ];

        return response()->json([
            'status' => true,
            'data'   => $packageData
        ]);
    }

    public function index()
    {
        // Eager load all related models: features, durationType, discounts, and priceType
        $packages = PackageMaster::with([
            'features',      // Eager load related features
            'durationType',  // Eager load the related duration type
            'discounts',     // Eager load related discounts
            'priceType'      // Eager load related price type
        ])->get();

        // Prepare the response with the required data, including Fea_Id, D_Id, and other related data
        $packages = $packages->map(function ($package) {
            return [
                'PackageId'      => $package->PackageId,
                'PackageName'    => $package->PackageName,
                'Description'    => $package->Description,
                'Status'         => $package->Status,
                'MaxUsers'       => $package->MaxUsers,
                'Duration_Value' => $package->Duration_Value,
                'DT_Id'          => $package->DT_Id,
                'Dis_Merge'      => $package->Dis_Merge,
                'PVT_Id'         => $package->PVT_Id,
                'Final_Price'    => $package->Final_Price,
                'Total_Duration' => $package->Total_Duration,
                'Fea_Id'         => $package->features->pluck('Package_Fea_Id')->toArray(), // Get only Fea_Id
                'D_Id'           => $package->discounts->pluck('D_Id')->toArray(), // Get only D_Id
                'created_at'     => $package->created_at,
                'updated_at'     => $package->updated_at,
                'features'       => $package->features,      // Include full features data
                'durationType'   => $package->durationType,  // Include full duration type data
                'discounts'      => $package->discounts,     // Include full discounts data
                'priceType'      => $package->priceType,     // Include full price type data
            ];
        });

        return response()->json([
            'status' => true,
            'data'   => $packages
        ]);
    }

    public function destroy($id)
    {
        $package = PackageMaster::findOrFail($id);
        $package->delete();
        return response()->json([
            'status' => true,
            'message' => 'Package deleted successfully!'
        ]);
    }

    private function getDurationFactor($durationType)
    {
        $factors = [
            'day' => 1,
            'month' => 30, // Approximate
            'year' => 365  // Approximate
        ];
        return $factors[strtolower($durationType)] ?? 1;
    }
}
