<?php

namespace App\Http\Controllers;

use App\Mail\ScheduleApprovedMail;
use App\Models\ScheduleDemo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;


class ScheduleDemoController extends Controller
{
    // 🔹 Create (Store)
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'sche_name' => 'required|string',
                'sche_email' => 'required|email',
                'sche_phone' => 'required|string',
                'sche_date' => 'required|date',
                'sche_time_slots' => 'required|string',
                'additional_note' => 'nullable|string',
            ]);

            // Handle validation failure
            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            // Add default sche_status value of 0 (pending)
            $validated['sche_status'] = 0;

            // Create the schedule
            $schedule = ScheduleDemo::create($validated);

            return response()->json([
                'status' => true,
                'message' => 'Schedule Demo request submitted successfully!',
                'data' => $schedule
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to submit schedule request.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Read (List all)
    public function index()
    {
        try {
            $schedules = ScheduleDemo::all();

            // Check if there are no schedules
            if ($schedules->isEmpty()) {
                return response()->json(['message' => 'No schedules found'], 404);
            }

            return response()->json($schedules);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to retrieve schedules.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 Read (Single)
    public function show($id)
    {
        try {
            $schedule = ScheduleDemo::find($id);

            if (!$schedule) {
                return response()->json(['message' => 'Schedule not found'], 404);
            }

            return response()->json($schedule);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to retrieve schedule.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $schedule = ScheduleDemo::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'sche_name'       => 'sometimes|required|string',
                'sche_email'      => 'sometimes|required|email',
                'sche_phone'      => 'sometimes|required|string',
                'sche_date'       => 'sometimes|required|date',
                'sche_time_slots' => 'sometimes|required|string',
                'additional_note' => 'nullable|string',
                'sche_status'     => 'nullable|in:0,1,2',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status'  => false,
                    'message' => 'Validation failed.',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            $previousStatus = $schedule->sche_status;
            $schedule->update($validated);

            if ($previousStatus != 1 && $schedule->sche_status == 1) {
                Mail::to($schedule->sche_email)->send(new ScheduleApprovedMail($schedule));
            }

            return response()->json([
                'status'  => true,
                'message' => 'Schedule updated successfully.',
                'data'    => $schedule
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status'  => false,
                'message' => 'Update failed.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }
    // public function update(Request $request, $id)
    // {
    //     try {
    //         $schedule = ScheduleDemo::findOrFail($id);

    //         $validator = Validator::make($request->all(), [
    //             'sche_name'         => 'sometimes|required|string',
    //             'sche_email'        => 'sometimes|required|email',
    //             'sche_phone'        => 'sometimes|required|string',
    //             'sche_date'         => 'sometimes|required|date',
    //             'sche_time_slots'   => 'sometimes|required|string',
    //             'additional_note'   => 'nullable|string',
    //             'sche_status'       => 'nullable|in:0,1,2',
    //         ]);

    //         if ($validator->fails()) {
    //             return response()->json([
    //                 'status'  => false,
    //                 'message' => 'Validation failed.',
    //                 'errors'  => $validator->errors()
    //             ], 422);
    //         }

    //         $validated = $validator->validated();

    //         $previousStatus = $schedule->sche_status;
    //         $schedule->update($validated);

    //         if ($previousStatus != 1 && $schedule->sche_status == 1) {
    //             Mail::to($schedule->sche_email)->send(new ScheduleApprovedMail($schedule));
    //         }

    //         return response()->json([
    //             'status'  => true,
    //             'message' => 'Schedule updated successfully.',
    //             'data'    => $schedule
    //         ]);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status'  => false,
    //             'message' => 'Update failed.',
    //             'error'   => $e->getMessage()
    //         ], 500);
    //     }
    // }

    // 🔹 Delete
    public function destroy($id)
    {
        try {
            $schedule = ScheduleDemo::find($id);

            if (!$schedule) {
                return response()->json(['message' => 'Schedule not found'], 404);
            }

            $schedule->delete();

            return response()->json([
                'status' => true,
                'message' => 'Schedule deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete schedule.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
