<?php

namespace App\Http\Controllers;

use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class TagController extends Controller
{
    public function index()
    {
        try {
            $tags = Tag::all();
            return response()->json($tags, 200);
        } catch (\Exception $e) {
            Log::error('Failed to fetch tags: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Failed to fetch tags.', 'details' => $e->getMessage()], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Tag_Name' => 'required|string|max:255|unique:tags,Tag_Name',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $tag = Tag::create([
                'Tag_Name' => $request->Tag_Name,
            ]);

            return response()->json($tag, 201);
        } catch (\Exception $e) {
            Log::error('Failed to create tag: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Failed to create tag.', 'details' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        try {
            $tag = Tag::find($id);
            if (!$tag) {
                return response()->json(['error' => 'Tag not found.'], 404);
            }
            return response()->json($tag, 200);
        } catch (\Exception $e) {
            Log::error('Failed to fetch tag: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Failed to fetch tag.', 'details' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $tag = Tag::find($id);
        if (!$tag) {
            return response()->json(['error' => 'Tag not found.'], 404);
        }

        $validator = Validator::make($request->all(), [
            'Tag_Name' => 'sometimes|required|string|max:255|unique:tags,Tag_Name,' . $id . ',Tag_Id',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $tag->update([
                'Tag_Name' => $request->Tag_Name ?? $tag->Tag_Name,
            ]);

            return response()->json($tag, 200);
        } catch (\Exception $e) {
            Log::error('Failed to update tag: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Failed to update tag.', 'details' => $e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        $tag = Tag::find($id);
        if (!$tag) {
            return response()->json(['error' => 'Tag not found.'], 404);
        }

        try {
            $tag->delete();
            return response()->json(['message' => 'Tag deleted successfully.'], 200);
        } catch (\Exception $e) {
            Log::error('Failed to delete tag: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json(['error' => 'Failed to delete tag.', 'details' => $e->getMessage()], 500);
        }
    }
}
