<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\ComPackage;
use App\Models\PackageMaster;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserMasterController extends Controller
{
    public function index()
    {
        $users = UserMaster::with([
            'role'
        ])->get();

        $data = $users->map(function ($user) {
            $package = $user->package;

            return [
                'U_Id' => $user->U_Id,
                'U_Name' => $user->U_Name,
                'U_Email' => $user->U_Email,
                'U_Mobile' => $user->U_Mobile,
                'PAN_No' => $user->PAN_No,
                'Com_Regi_No' => $user->Com_Regi_No,
                'Short_CommName' => $user->Short_CommName,
                'Comm_Name' => $user->Comm_Name,
                'Comm_Id' => $user->Comm_Id,
                'Role_Id' => $user->Role_Id,
                'PackageId' => $user->PackageId,
                'U_Status' => $user->U_Status,
                'Middle_Name' => $user->Middle_Name,
                'Last_Name' => $user->Last_Name,
                'Address' => $user->Address,
                'City' => $user->City,
                'State' => $user->State,
                'Country' => $user->Country,
                'Zipcode' => $user->Zipcode,
                'Gender' => $user->Gender,
                'Occupation' => $user->Occupation,
                'Marital_Status' => $user->Marital_Status,
                'Education_School' => $user->Education_School,
                'Education_College' => $user->Education_College,
                'P_Image_Relative' => $user->P_Image ? str_replace(asset(''), '', $user->P_Image) : null, // Relative path: profiles/image.png
                'P_Image_URL' => $user->P_Image, // Full URL: http://localhost:8000/profiles/image.png
                'School_Name' => $user->School_Name,
                'College_Name' => $user->College_Name,
                'Occupation_Address' => $user->Occupation_Address,
                'DOB' => $user->DOB,
                'deleted_at' => $user->deleted_at,
                'created_at' => $user->created_at,
                'updated_at' => $user->updated_at,
                'role' => $user->role,
                'package' => $package ? array_merge(
                    $package->toArray(),
                    [
                        'Fea_Id' => $package->features->pluck('Package_Fea_Id')->toArray(),
                        'D_Id' => $package->discounts->pluck('D_Id')->toArray(),
                    ]
                ) : null
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200);
    }

    public function store(Request $request)
    {
        try {
            // Set default Role_Id = 3 if not provided
            if (!$request->has('Role_Id')) {
                $request->merge(['Role_Id' => 3]);
            }

            // Set default PackageId = 1 if not provided
            if (!$request->has('PackageId')) {
                $request->merge(['PackageId' => 1]);
            }

            $data = $request->all();

            // Check if only one SuperAdmin exists
            if ($request->Role_Id == 1) {
                $superAdminExists = UserMaster::where('Role_Id', 1)->exists();
                if ($superAdminExists) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'SuperAdmin already exists. Only one SuperAdmin is allowed.',
                    ], 403);
                }
            }

            // Validation rules
            $validator = Validator::make($data, [
                'U_Name' => 'required|string',
                'U_Email' => 'required|email|max:255|unique:user_masters,U_Email',
                'U_Password' => 'required|string|min:8',
                'U_Mobile' => 'required|digits:10|unique:user_masters,U_Mobile',
                'Role_Id' => 'required|integer|exists:role_masters,Role_Id',
                'PackageId' => [
                    'required',
                    'integer',
                    Rule::exists('package_masters', 'PackageId')->where(function ($query) use ($data) {
                        if (isset($data['Role_Id']) && $data['Role_Id'] == 2) {
                            $query->where('Total_Duration', '>', 0);
                        }
                    }),
                ],
                'Short_CommName' => 'required_if:Role_Id,2|string',
                'Comm_Name' => 'required_if:Role_Id,2|string',
                'PAN_No' => 'required_if:Role_Id,2|string|max:20',
                'Com_Regi_No' => 'nullable|string|max:20',
                'Comm_Id' => [
                    'required_if:Role_Id,3',
                    'string',
                    Rule::exists('user_masters', 'Comm_Id')->where(function ($query) use ($data) {
                        if (isset($data['Role_Id']) && $data['Role_Id'] == 3) {
                            $query->where('Role_Id', 2);
                        }
                    }),
                ],
                'P_Image' => 'nullable|image|mimes:jpeg,jpg,png,gif,bmp,webp',
                'Middle_Name' => 'nullable|string',
                'Last_Name' => 'nullable|string',
                'Address' => 'nullable|string',
                'City' => 'nullable|string',
                'State' => 'nullable|string',
                'Country' => 'nullable|string',
                'Zipcode' => 'nullable|string',
                'Gender' => 'nullable|in:Male,Female,Other',
                'Occupation' => 'nullable|string',
                'Marital_Status' => 'nullable|in:Married,Single',
                'Education_School' => 'nullable|numeric|between:0,100',
                'Education_College' => 'nullable|numeric|between:0,10',
                'School_Name' => 'nullable|string',
                'College_Name' => 'nullable|string',
                'Occupation_Address' => 'nullable|string',
                'DOB' => 'nullable|date',
            ], [
                'Short_CommName.required_if' => 'The Short Community Name is required when Role is Admin.',
                'Comm_Name.required_if' => 'The Community Name is required when Role is Admin.',
                'Comm_Id.required_if' => 'The Community ID is required when Role is Community Member.',
                'Comm_Id.exists' => 'The selected Community ID is invalid or not linked to an Admin.',
                'PackageId.exists' => 'The selected Package ID does not exist or has an invalid duration.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Format Short_CommName as 'U_Name@Short_CommName' if Role_Id = 2
            if ($data['Role_Id'] == 2 && isset($data['Short_CommName'], $data['U_Name'])) {
                $data['Short_CommName'] = $data['U_Name'] . '@' . $data['Short_CommName'];
            }

            // Check if committee admin's U_Status is active (U_Status = 1) for Role_Id = 3
            if (isset($data['Comm_Id']) && $data['Role_Id'] == 3) {
                $committee = UserMaster::where('Comm_Id', $data['Comm_Id'])
                    ->where('Role_Id', 2)
                    ->first();

                if (!$committee || !in_array($committee->U_Status, [1])) {
                    return response()->json([
                        'status' => 'error',
                        'errors' => [
                            'Comm_Id' => [
                                'The selected Community ID is associated with an inactive committee. Please contact your community admin for assistance or activate the committee.'
                            ]
                        ]
                    ], 422);
                }
            }

            // Handle image upload
            if ($request->hasFile('P_Image')) {
                $image = $request->file('P_Image');
                $filename = time() . '_' . $image->getClientOriginalName();
                $image->move(public_path('profiles'), $filename);
                $relativePath = 'profiles/' . $filename;
                $data['P_Image'] = asset($relativePath); // Save full URL to database
            }

            // Hash password and set default status
            $data['U_Password'] = Hash::make($data['U_Password']);
            $data['U_Status'] = 0;

            // Generate Comm_Id if Admin (Role_Id = 2)
            if ($data['Role_Id'] == 2) {
                $data['Comm_Id'] = rand(100000, 999999);
            }

            // Set St_Date and End_Date for Role_Id = 2
            if ($data['Role_Id'] == 2) {
                $package = PackageMaster::find($data['PackageId']);
                if (!$package || !$package->Total_Duration || $package->Total_Duration <= 0) {
                    return response()->json([
                        'status' => 'error',
                        'errors' => ['PackageId' => ['Invalid package or no valid duration specified.']],
                    ], 422);
                }
                $data['St_Date'] = now()->toDateString();
                $data['End_Date'] = now()->addDays($package->Total_Duration)->toDateString();
            }

            // Create user
            $user = UserMaster::create($data);

            // Fetch discounts for the selected PackageId
            $package = PackageMaster::find($data['PackageId']);
            if ($package) {
                $discountIds = $package->discounts->pluck('D_Id')->toArray();
            } else {
                $discountIds = [];
            }

            // Store in com_packages for Role_Id = 2
            if ($data['Role_Id'] == 2) {
                ComPackage::create([
                    'U_Id' => $user->U_Id,
                    'St_Date' => $data['St_Date'],
                    'End_Date' => $data['End_Date'],
                    'U_Name' => $data['U_Name'],
                    'PackageId' => $package->PackageId,
                    'PackageName' => $package->PackageName ?? null,
                    'Package_Description' => $package->Description ?? null,
                    'Package_Status' => $package->Status ?? null,
                    'Package_MaxUsers' => $package->MaxUsers ?? null,
                    'DT_Id' => $package->DT_Id ?? null,
                    'D_Id' => json_encode($discountIds), // Store as JSON string
                    'Dis_Merge' => $package->Dis_Merge ?? null,
                    'PVT_Id' => $package->PVT_Id ?? null,
                    'Final_Price' => $package->Final_Price ?? null,
                    'Total_Duration' => $package->Total_Duration ?? null,
                ]);
            }

            // Prepare response with relative path and full URL
            $responseData = $user->toArray();
            // if ($user->P_Image) {
            //     $responseData['P_Image_Relative'] = str_replace(asset(''), '', $user->P_Image); // Relative path
            //     $responseData['P_Image_URL'] = $user->P_Image; // Full URL
            // }

            return response()->json([
                'status' => 'success',
                'message' => 'User registered successfully.',
                'data' => $responseData,
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to register user.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function show($id)
    {
        $user = UserMaster::with([
            'role'
        ])->find($id);

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not found',
            ], 404);
        }

        $package = $user->package;

        $data = [
            'U_Id' => $user->U_Id,
            'U_Name' => $user->U_Name,
            'U_Email' => $user->U_Email,
            'U_Mobile' => $user->U_Mobile,
            'PAN_No' => $user->PAN_No,
            'Com_Regi_No' => $user->Com_Regi_No,
            'Short_CommName' => $user->Short_CommName,
            'Comm_Name' => $user->Comm_Name,
            'Comm_Id' => $user->Comm_Id,
            'Role_Id' => $user->Role_Id,
            'PackageId' => $user->PackageId,
            'U_Status' => $user->U_Status,
            'Middle_Name' => $user->Middle_Name,
            'Last_Name' => $user->Last_Name,
            'Address' => $user->Address,
            'City' => $user->City,
            'State' => $user->State,
            'Country' => $user->Country,
            'Zipcode' => $user->Zipcode,
            'Gender' => $user->Gender,
            'Occupation' => $user->Occupation,
            'Marital_Status' => $user->Marital_Status,
            'Education_School' => $user->Education_School,
            'Education_College' => $user->Education_College,
            'P_Image_Relative' => $user->P_Image ? str_replace(asset(''), '', $user->P_Image) : null, // Relative path
            'P_Image_URL' => $user->P_Image, // Full URL
            'School_Name' => $user->School_Name,
            'College_Name' => $user->College_Name,
            'Occupation_Address' => $user->Occupation_Address,
            'DOB' => $user->DOB,
            'deleted_at' => $user->deleted_at,
            'created_at' => $user->created_at,
            'updated_at' => $user->updated_at,
            'role' => $user->role,
            'package' => $package ? array_merge(
                $package->toArray(),
                [
                    'Fea_Id' => $package->features->pluck('Package_Fea_Id')->toArray(),
                    'D_Id' => $package->discounts->pluck('D_Id')->toArray(),
                ]
            ) : null
        ];

        return response()->json([
            'status' => 'success',
            'data' => $data,
        ], 200);
    }

    public function update(Request $request, $id)
    {
        try {
            $user = UserMaster::find($id);

            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found.',
                ], 404);
            }

            $data = $request->all();

            $validator = Validator::make($data, [
                'U_Name' => 'sometimes|string',
                'U_Email' => 'sometimes|email|max:255|unique:user_masters,U_Email,' . $user->U_Id . ',U_Id',
                'U_Password' => 'sometimes|nullable|string|min:8',
                'U_Mobile' => 'sometimes|digits:10|unique:user_masters,U_Mobile,' . $user->U_Id . ',U_Id',
                'Role_Id' => 'sometimes|integer|exists:role_masters,Role_Id',
                'PackageId' => [
                    'sometimes',
                    'integer',
                    Rule::exists('package_masters', 'PackageId')->where(function ($query) use ($data, $user) {
                        if ((isset($data['Role_Id']) && $data['Role_Id'] == 2) || $user->Role_Id == 2) {
                            $query->where('Total_Duration', '>', 0);
                        }
                    }),
                ],
                'Short_CommName' => 'sometimes|string',
                'Comm_Name' => 'sometimes|string',
                'PAN_No' => 'nullable|string|max:20',
                'Com_Regi_No' => 'nullable|string|max:20',
                'Comm_Id' => [
                    'sometimes',
                    'string',
                    Rule::exists('user_masters', 'Comm_Id')->where(function ($query) use ($data, $user) {
                        if ((isset($data['Role_Id']) && $data['Role_Id'] == 3) || $user->Role_Id == 3) {
                            $query->where('Role_Id', 2);
                        }
                    }),
                ],
                'P_Image' => 'sometimes|string', // Base64 expected
                'Middle_Name' => 'nullable|string',
                'Last_Name' => 'nullable|string',
                'Address' => 'nullable|string',
                'City' => 'nullable|string',
                'State' => 'nullable|string',
                'Country' => 'nullable|string',
                'Zipcode' => 'nullable|string',
                'Gender' => 'nullable|in:Male,Female,Other',
                'Occupation' => 'nullable|string',
                'Marital_Status' => 'nullable|in:Married,Single',
                'Education_School' => 'nullable|numeric|between:0,100',
                'Education_College' => 'nullable|numeric|between:0,10',
                'School_Name' => 'nullable|string',
                'College_Name' => 'nullable|string',
                'Occupation_Address' => 'nullable|string',
                'DOB' => 'nullable|date',
            ], [
                'Comm_Id.exists' => 'The selected Community ID is invalid or does not belong to an admin.',
                'PackageId.exists' => 'The selected Package ID does not exist or has an invalid duration.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'errors' => $validator->errors(),
                ], 422);
            }

            if (isset($data['Role_Id']) && $data['Role_Id'] == 1 && $user->Role_Id != 1) {
                $superAdminExists = UserMaster::where('Role_Id', 1)
                    ->where('U_Id', '!=', $user->U_Id)
                    ->exists();

                if ($superAdminExists) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'SuperAdmin already exists. Only one SuperAdmin is allowed.',
                    ], 403);
                }
            }

            // Insert com_packages if PackageId is present
            if (isset($data['PackageId']) && ($user->Role_Id == 2 || (isset($data['Role_Id']) && $data['Role_Id'] == 2))) {
                $package = PackageMaster::find($data['PackageId']);
                if (!$package || !$package->Total_Duration || $package->Total_Duration <= 0) {
                    return response()->json([
                        'status' => 'error',
                        'errors' => ['PackageId' => ['Invalid package or no valid duration specified.']],
                    ], 422);
                }

                $data['St_Date'] = now()->toDateString();
                $data['End_Date'] = now()->addDays($package->Total_Duration)->toDateString();

                ComPackage::create([
                    'U_Id' => $user->U_Id,
                    'St_Date' => $data['St_Date'],
                    'End_Date' => $data['End_Date'],
                    'U_Name' => $data['U_Name'] ?? $user->U_Name,
                    'PackageId' => $package->PackageId,
                    'PackageName' => $package->PackageName ?? null,
                    'Package_Description' => $package->Description ?? null,
                    'Package_Status' => $package->Status ?? null,
                    'Package_MaxUsers' => $package->MaxUsers ?? null,
                    'DT_Id' => $package->DT_Id ?? null,
                    'D_Id' => json_encode($package->discounts->pluck('D_Id')->toArray()), // Store as JSON string
                    'Dis_Merge' => $package->Dis_Merge ?? null,
                    'PVT_Id' => $package->PVT_Id ?? null,
                    'Final_Price' => $package->Final_Price ?? null,
                    'Total_Duration' => $package->Total_Duration ?? null,
                ]);
            }

            // Base64 image upload
            if (!empty($data['P_Image']) && strpos($data['P_Image'], 'data:image') === 0) {
                list($type, $imageData) = explode(';', $data['P_Image']);
                list(, $imageData) = explode(',', $imageData);
                $imageName = time() . '_' . $user->U_Id . '.png';
                $imagePath = public_path('profiles/' . $imageName);
                file_put_contents($imagePath, base64_decode($imageData));
                $relativePath = 'profiles/' . $imageName;
                $data['P_Image'] = asset($relativePath); // Save full URL to database
            } else {
                unset($data['P_Image']); // If not valid base64 or empty, skip
            }

            // Password hashing
            if (!empty($data['U_Password'])) {
                $data['U_Password'] = Hash::make($data['U_Password']);
            } else {
                unset($data['U_Password']);
            }

            // Short_CommName setup for Role_Id 2
            if ((isset($data['Role_Id']) && $data['Role_Id'] == 2) || $user->Role_Id == 2) {
                if (isset($data['Short_CommName'], $data['U_Name'])) {
                    $data['Short_CommName'] = $data['U_Name'] . '@' . $data['Short_CommName'];
                } elseif (isset($data['Short_CommName'])) {
                    $data['Short_CommName'] = $user->U_Name . '@' . $data['Short_CommName'];
                }
            }

            // Final filtering and updating
            $filteredData = collect($data)->filter(function ($value) {
                return $value !== null && $value !== '';
            });

            $user->fill($filteredData->toArray());
            $user->save();

            // Prepare response with relative path and full URL
            $responseData = $user->toArray();
            if ($user->P_Image) {
                $responseData['P_Image_Relative'] = str_replace(asset(''), '', $user->P_Image); // Relative path
                $responseData['P_Image_URL'] = $user->P_Image; // Full URL
            }

            return response()->json([
                'status' => 'success',
                'message' => 'User updated successfully.',
                'data' => $responseData,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update user.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function destroy($id)
    {
        $user = UserMaster::find($id);

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not found.',
            ], 404);
        }

        // Delete image file
        if ($user->P_Image && file_exists(public_path(str_replace(asset(''), '', $user->P_Image)))) {
            unlink(public_path(str_replace(asset(''), '', $user->P_Image)));
        }

        $user->delete(); // Performs soft delete

        return response()->json([
            'status' => 'success',
            'message' => 'User soft deleted successfully.',
        ]);
    }

    public function getDeletedUsers()
    {
        try {
            $users = UserMaster::onlyTrashed()->get();

            if ($users->isEmpty()) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'No soft deleted users found',
                    'data' => []
                ], 200);
            }

            // Convert image paths to relative and full URL
            $users->transform(function ($user) {
                if ($user->P_Image) {
                    $user->P_Image_Relative = str_replace(asset(''), '', $user->P_Image); // Relative path
                    $user->P_Image_URL = $user->P_Image; // Full URL
                }
                return $user;
            });

            return response()->json([
                'status' => 'success',
                'message' => 'Soft deleted users fetched successfully',
                'data' => $users
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch soft deleted users',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
